<?php

require_once 'vendor/autoload.php';


require_once 'model/MongoDBConnection.php';
require_once 'controller/MailHelper.php';

require_once 'controller/JSONHelper.php';

class NhanVien {
    private $collection;

    
    private function generateRandomPassword(int $length = 8){
        // bộ ký tự dùng để random
        $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $charactersLength = strlen($characters);
        $password = '';

        for ($i = 0; $i < $length; $i++) {
            // random vị trí 0 -> $charactersLength - 1
            $index = random_int(0, $charactersLength - 1);
            $password .= $characters[$index];
        }

        return $password;
    }

    // Constructor để lấy collection 'nhan_vien'
    public function __construct() {
        // Lấy kết nối MongoDB
        $mongoDB = MongoDBConnection::getInstance();
        $this->collection = $mongoDB->getCollection('nhan_vien');
    }

    
    public function addNhanVien($data){
        
        $data["mat_khau"] = $this->generateRandomPassword();

        $dataMail = [
            "username" => $data["ten_nhan_vien"],
            "password" => $data["mat_khau"],
            "note"     => "Hãy đổi mật khẩu sau khi đăng nhập."
        ];

        MailHelper::sendNewPasswordEmail($data['email'], $dataMail);

        $data["mat_khau"] =  password_hash($data["mat_khau"], PASSWORD_DEFAULT);

        $insertResult = $this->collection->insertOne($data);

        $data["_id"] = (string) $insertResult->getInsertedId();
        $data["active"] = true;

        return (string) $insertResult->getInsertedId(); 
    }

    public function getAllNhanVien($page = 1){
        $page = (int)$page;
        if ($page < 1) {
            $page = 1;
        }

        // Mặc định 20 nhân viên / trang
        $limit = 15;
        $skip  = ($page - 1) * $limit;

        // Chỉ sort theo "active"
        $options = [
            'sort'  => ['active' => -1], 
            'skip'  => $skip,
            'limit' => $limit
        ];

        // Lấy dữ liệu theo trang
        $cursor = $this->collection->find([], $options);
        $items  = iterator_to_array($cursor);

        // Tổng số nhân viên
        $totalItems = $this->collection->countDocuments();
        $totalPages = (int) ceil($totalItems / $limit);

        // Xoá ObjectId
        $items = JSONHelper::removeOid($items);

        return [
            'list_nhan_vien' => $items,
            'pagination' => [
                'page'        => $page,
                'total_items' => $totalItems,
                'total_pages' => $totalPages
            ]
        ];
    }


    public function loginNhanVien($data) {

        // 2. Tìm nhân viên theo username OR email OR số điện thoại
        $cursor = $this->collection->findOne([
            '$or' => [
                ['email'         => $data["username"]],
                ['so_dien_thoai' => $data["username"]]
            ]
        ]);

        if (!$cursor) {
            // Không tìm thấy tài khoản
            return null;
        }

        if ($cursor["active"] == false )
            return null;

        $info = $cursor->getArrayCopy();

        // 3. Kiểm tra mật khẩu có tồn tại không
        if (empty($info['mat_khau'])) {
            return null; // Không có mật khẩu -> fail
        }

       
        // 4. So sánh mật khẩu
        if (!password_verify($data["mat_khau"], $info['mat_khau'])) {
            return null; // Sai mật khẩu
        }

        // 5. Login thành công -> xóa mật khẩu trước khi trả về
        unset($info['mat_khau']);

        return  JSONHelper::removeOid($info); // trả về info nhân viên
    }

    
    //function reset password
    public function resetPassWord($data, $mode){

        $data["mat_khau"] = $this->generateRandomPassword();

        $dataMail = [
            "username" => $data["ten_nhan_vien"],
            "password" => $data["mat_khau"],
            "note"     => "Hãy đổi mật khẩu sau khi đăng nhập."
        ];

        MailHelper::sendNewPasswordEmail($data['email'], $dataMail);

        unset($data["ten_nhan_vien"]);
        unset($data["email"]);

        return ($this->updatePassword($data, $mode));
        
    }

    public function updatePassword($data, $mode = "change"){
        // Lấy nhân viên
        $cursor = $this->collection->findOne([
            '_id' => new MongoDB\BSON\ObjectId($data['_id'])
        ]);

        
        if (!$cursor) {
            return false;
        }

        $info = $cursor->getArrayCopy();

        // ===========================
        // MODE: change (đổi mật khẩu)
        // ===========================
        if ($mode === "change") {
            // Chỉ kiểm tra mật khẩu cũ
            if (!password_verify($data['mat_khau_cu'], $info['mat_khau'])) {
                return false;
            }

            $newHash = password_hash($data['mat_khau'], PASSWORD_DEFAULT);

            $update = $this->collection->updateOne(
                ['_id' => $info['_id']],
                ['$set' => ['mat_khau' => $newHash]]
            );

            return  $update->getModifiedCount();
        }

        // ===========================
        // MODE: forgot (quên mật khẩu)
        // ===========================
        if ($mode === "forgot") {

   
            $newHash = password_hash($data["mat_khau"], PASSWORD_DEFAULT);

            $updateResult = $this->collection->updateOne(
                ['_id' => $info['_id']],
                ['$set' => ['mat_khau' => $newHash]]
            );

            return $updateResult->getMatchedCount();
        }

        return 0;
    }

    // Update: Cập nhật thông tin nhân viên
    public function updateNhanVien($id, $data) {
        
        //Sử dụng replaceOne để ghi đè mọi dữ liệu 
        $updateResult = $this->collection->updateOne(
            ['_id' => new MongoDB\BSON\ObjectId($id)],
            ['$set' => $data]
        );

        return $updateResult->getModifiedCount(); // Số bản ghi bị thay đổi
    }
    
}
?>