<?php

require_once 'vendor/autoload.php';

require_once 'model/MongoDBConnection.php';
require_once 'model/ThuCung.php';

require_once 'controller/JSONHelper.php';
require_once 'controller/TimeHelper.php';

class KhachHang {
    private $collection;

    // Constructor để lấy collection 'khach_hang'
    public function __construct() {
        // Lấy kết nối MongoDB
        $mongoDB = MongoDBConnection::getInstance();
        $this->collection = $mongoDB->getCollection('khach_hang');
    }

    // Lấy khách hàng theo id full-detail cho trang khách hàng, ngược lại cho những trang tìm kiếm khác
    public function getKhachHangWithPets($id, $type ="full-detail"){

        if ($type == "full-detail"){
            $cursor = $this->collection->findOne(['_id' => new MongoDB\BSON\ObjectId($id)]);
        } else {
            $cursor = $this->collection->findOne(
                ['_id' => new MongoDB\BSON\ObjectId($id)],
                [
                    'projection' => [
                        '_id' => 1,
                        'ten_khach_hang' => 1,
                        'so_dien_thoai' => 1,
                        'pets' => 1
                    ]
                ]
            );
        }
        
        
        if (!$cursor)
            return null;

        // Chuyển về mảng dễ xử lý
        $result = $cursor->getArrayCopy();
        $thuCung = new ThuCung();

        // Nếu có pets, tìm từng pet theo ObjectId
        if (isset($result['pets']) && is_iterable($result['pets'])) {
            $pets = [];
            foreach ($result['pets'] as $petId) {
                $pet = $thuCung->getThuCung((string) $petId, $type);
                if ($pet) {
                    $pets[] = $pet->getArrayCopy();
                }
            }
            $result['pets'] = $pets;
        } else {
            $result['pets'] = [];
        }

        
        return JSONHelper::removeOid($result);
    }

    // Create: Thêm khách hàng mới
    public function addKhachHang($data) {
        $insertResult = $this->collection->insertOne($data);
        return (string) $insertResult->getInsertedId(); 
    }

    // Update: Cập nhật thông tin khách hàng
    public function updateKhachHang($id, $data, $type = 'update') {
        $updateData = [];
        
        // Nếu là cập nhật thông tin cơ bản của khách hàng
        if ($type == 'update') {
            $updateData = ['$set' => $data]; // Sử dụng $set để sửa thông tin khách hàng
        }

        // Nếu thêm pet vào danh sách thú cưng
        if ($type == 'add_pet' ) {
            $updateData = ['$push' => ['pets' => $data]]; // Thêm pet vào danh sách thú cưng
        }

        // Nếu là loại bỏ pet khỏi khách hàng
        if ($type == 'remove_pet') {
            $updateData = ['$pull' => ['pets' => new MongoDB\BSON\ObjectId($data)]]; // Loại bỏ pet khỏi danh sách thú cưng
        }

        // Cập nhật thông tin khách hàng
        $updateResult = $this->collection->updateOne(
            ['_id' => new MongoDB\BSON\ObjectId($id)],
            $updateData
        );
        
        return $updateResult->getModifiedCount(); // Trả về số bản ghi bị thay đổi
    }

    // DELETE: Xóa khách hàng
    public function deleteKhachHang($id) {
        $deleteResult = $this->collection->deleteOne(['_id' => new MongoDB\BSON\ObjectId($id)]);
        return $deleteResult->getDeletedCount(); // Trả về số bản ghi đã bị xóa
    }

    // Lấy tất cả khách hàng kèm pets, phân trang + tìm kiếm theo kw
    public function getAllKhachHang(int $page = 1, string $kw = ""){
        $limit = 15;

        if ($page < 1) $page = 1;
        $skip = ($page - 1) * $limit;

        // ---- 1. Tạo điều kiện tìm kiếm ----
        $filter = [];

        if ($kw !== "") {
            // Regex không phân biệt hoa thường
            $regex = new MongoDB\BSON\Regex($kw, 'i');

            $filter = [
                '$or' => [
                    ['ten_khach_hang' => $regex],
                    ['so_dien_thoai'   => $regex]
                ]
            ];
        }

        // ---- 2. Truy vấn danh sách khách theo trang ----
        $cursor = $this->collection->find(
            $filter,
            [
                'skip'  => $skip,
                'limit' => $limit,
                'sort'  => ['_id' => -1], // mới nhất trước
            ]
        );

        // ---- 3. Dùng lại getKhachHangWithPets() cho từng khách ----
        $items = [];
        foreach ($cursor as $doc) {
            $idKhachHang = (string) $doc['_id'];

            $khachHang = $this->getKhachHangWithPets($idKhachHang, "full-detail");

            if ($khachHang !== null) {
                $items[] = $khachHang;
            }
        }

        // ---- 4. Tổng số document để tính tổng trang ----
        $total = $this->collection->countDocuments($filter);
        $totalPages = (int) ceil($total / $limit);

        // ---- 5. Trả kết quả ----
        return [
            'list_khach_hang' => JSONHelper::removeOid($items),
            'pagination' => [
                'page'        => $page,
                'total_items' => $total,
                'total_pages' => $totalPages
            ]
        ];
    }    

    //Kiểm tra tồn tại số điện thoại 
    public function isExistsSDT($soDienThoai){

        $soDienThoai = trim($soDienThoai);
        
        $cursor = $this->collection->findOne(['so_dien_thoai' => $soDienThoai]);

        if ($cursor){
            return true;
        } else {
            return false;
        }

    }


    // Function to search customers by name or phone number and include their pets
    public function autoCompleteKhachHang($searchQuery) {

        if ($searchQuery == "")
            return null;

        // Using regular expression for case-insensitive search
        $regex = new MongoDB\BSON\Regex($searchQuery, 'i');
        
        // Tìm khách hàng bằng tên hoặc số điện thoại
        $cursor = $this->collection->aggregate([
            [
                '$match' => [
                    '$or' => [
                        ['ten_khach_hang' => $regex], 
                        ['so_dien_thoai' => $regex] 
                    ]
                ]
            ],
            [
                '$project' => [
                    '_id' => 1, 
                    'ten_khach_hang' => 1, 
                    'so_dien_thoai' => 1 
                ]
            ],
            [
                '$limit' => 5 
            ]
        ]);

        
        $result = JSONHelper::removeOid(iterator_to_array($cursor));

        return $result; 
    }
}