<?php
require_once 'vendor/autoload.php';

require_once 'model/MongoDBConnection.php';
require_once 'model/ChiNhanh.php';
require_once 'model/Inventory.php';

require_once 'controller/JSONHelper.php';

class HangHoa{
    private $collection;

    // Constructor để lấy collection 'bill'
    public function __construct() {
        // Lấy kết nối MongoDB
        $mongoDB = MongoDBConnection::getInstance();
        $this->collection = $mongoDB->getCollection('hang_hoa');
    }

    //Sử dụng để kèm thêm Inventory
    private function attachInventory($data, $type="add"){

        $chiNhanh = new ChiNhanh();
        $listChiNhanh = $chiNhanh->getAllChiNhanh();
        $inventory = new Inventory();

        if (is_array($listChiNhanh)) {
            foreach ($listChiNhanh as $cn) {

                
                if (!isset($cn['_id'])) {
                    continue; // bỏ qua nếu thiếu id_chi_nhanh
                }

                // Nếu muốn lưu dạng ObjectId trong inventory:
                $inventoryData = [
                    'id_hang_hoa'  => new MongoDB\BSON\ObjectId($data["_id"]),                  
                    'id_chi_nhanh' => new MongoDB\BSON\ObjectId($cn["_id"]),       
                    'ton_kho'      => 0,                              // mặc định 0
                    'ton_toi_thieu' => $data["ton_toi_thieu"],
                    'ten_hang_hoa' => $data["ten_hang_hoa"],
                    'ten_don_vi' => $data["ten_don_vi"]
                ];
   
                if ($type =="add"){
                    $inventory->addInventory($inventoryData);
                    
                } elseif ($type == "update"){
                    unset($inventoryData["ton_kho"]);
                    $inventoryItem = $inventory->getInventory((string)$cn["_id"], $data["_id"]); 
               
                    $inventory->updateInventory($inventoryItem["_id"], $inventoryData);
                }
                
            }
        }
    }

    // Create: Thêm hàng hoá mới
    public function addHangHoa($data) {
        $insertResult = $this->collection->insertOne($data);
        $result = (string) $insertResult->getInsertedId(); // Trả về _id của hàng hoá mới
        $data["_id"] = $result;

        $this->attachInventory($data);
        return $result;
    }

    // Update: Cập nhật thông tin hàng hoá 
    public function updateHangHoa($id, $data) {
        
        //Sử dụng replaceOne để ghi đè mọi dữ liệu 
        $updateResult = $this->collection->replaceOne(
            ['_id' => new MongoDB\BSON\ObjectId($id)],
            $data
        );

        $data["_id"] = $id;
        $this->attachInventory($data, "update");

        return $updateResult->getModifiedCount(); // Số bản ghi bị thay đổi
    }

    // DELETE: Xóa hàng hoá
    public function deleteHangHoa($id) {
        $deleteResult = $this->collection->deleteOne(['_id' => new MongoDB\BSON\ObjectId($id)]);
        return $deleteResult->getDeletedCount(); // Trả về số bản ghi đã bị xóa
    }

    //autoComplete Hàng Hoá 
    public function autoCompleteHangHoa($searchQuery) {
        if ($searchQuery == "")
            return null;

        // Tạo regex tìm kiếm
        $regex = new MongoDB\BSON\Regex($searchQuery, 'i');

        // Tìm theo tên hàng hóa hoặc mã vạch
        $cursor = $this->collection->aggregate([
            [
                '$match' => [
                    '$or' => [
                        ['ten_hang_hoa' => $regex],
                        ['ma_vach'      => $regex]
                    ]
                ]
            ],
            [
                '$project' => [
                    '_id'          => 1,
                    'ten_hang_hoa' => 1
                ]
            ],
            [
                '$limit' => 5
            ]
        ]);

        // Convert về mảng + bỏ ObjectId
        return JSONHelper::removeOid(iterator_to_array($cursor));
    }

    //Lấy hàng hoá theo id full-detail cho trang hàng hoá, ngược lại cho những trang tìm kiếm khác
    public function getHangHoa($id, $type ="full-detail"){
        
        if ($type == "full-detail"){
            $cursor = $this->collection->findOne(['_id' => new MongoDB\BSON\ObjectId($id)]);
        } else {
            
            $cursor = $this->collection->findOne(
                ['_id' => new MongoDB\BSON\ObjectId($id)],
                [
                    'projection' => [
                        '_id' => 1,
                        'ten_hang_hoa' => 1,
                        'ten_don_vi' => 1,
                        'gia_ban' => 1
                    ]
                ]
            );
        }
        
        // Chuyển kết quả về mảng và loại bỏ ObjectId 
        return JSONHelper::removeOid($cursor);
    }

    public function getAllHangHoa($data){
        // Lấy page
        $page = isset($data["page"]) ? (int)$data["page"] : 1;
        if ($page < 1) $page = 1;

        // Lấy từ khoá keyword
        $kw = isset($data["kw"]) ? trim($data["kw"]) : "";

        // Lấy phân loại (EXACT MATCH)
        $phan_loai = isset($data["ten_phan_loai"]) ? trim($data["ten_phan_loai"]) : "";

        // Giới hạn & skip
        $limit = 15;
        $skip = ($page - 1) * $limit;

        // Tạo filter linh hoạt
        $filter = [];

        // Lọc theo keyword
        if ($kw !== "") {
            $filter["ten_hang_hoa"] = new MongoDB\BSON\Regex($kw, "i");
        }

        // Lọc theo phân loại — EXACT MATCH
        if ($phan_loai !== "") {
            $filter["ten_phan_loai"] = $phan_loai;
        }

        // Query MongoDB
        $cursor = $this->collection->find(
            $filter,
            [
                "limit" => $limit,
                "skip"  => $skip,
                "sort"  => ["_id" => -1],
                "projection" => [
                    "_id"           => 1,
                    "ten_hang_hoa"  => 1,
                    "ten_phan_loai" => 1,
                    "ten_don_vi"    => 1,
                    "gia_ban"       => 1
                ]
            ]
        );

        $items = iterator_to_array($cursor);

        // Đếm tổng bản ghi theo filter
        $totalItems = $this->collection->countDocuments($filter);
        $totalPages = (int)ceil($totalItems / $limit);

        // Return
        return [
            "list_hang_hoa" => JSONHelper::removeOid($items),
            "pagination" => [
                "page"        => $page,
                "total_items" => $totalItems,
                "total_pages" => $totalPages
            ]
        ];
    }


}
?>