<?php

require_once 'vendor/autoload.php';

require_once 'model/MongoDBConnection.php';
require_once 'model/HangHoa.php';

require_once 'controller/JSONHelper.php';

class DichVu{
    private $collection;

    // Constructor để lấy collection 'dich_vu'
    public function __construct() {
        // Lấy kết nối MongoDB
        $mongoDB = MongoDBConnection::getInstance();
        $this->collection = $mongoDB->getCollection('dich_vu');
    }

    private function processData($data){

        // 1. list_hang_hoa ở level 1
        if (isset($data['list_hang_hoa']) && is_array($data['list_hang_hoa'])) {
            foreach ($data['list_hang_hoa'] as &$item) {
                $item['id_hang_hoa'] =  new MongoDB\BSON\ObjectId($item['id_hang_hoa']);
            }
            unset($item); // tránh reference leak
        }

        // 2. list_hang_hoa bên trong từng option
        if (isset($data['options']) && is_array($data['options'])) {
            foreach ($data['options'] as &$option) {
                if (isset($option['list_hang_hoa']) && is_array($option['list_hang_hoa'])) {
                    foreach ($option['list_hang_hoa'] as &$item) {
                        $item['id_hang_hoa'] =  new MongoDB\BSON\ObjectId($item['id_hang_hoa']);
                    }
                    unset($item);
                }
            }
            unset($option);
        }

        return $data;
    }

    /**
     * Gắn thêm thông tin hàng hoá (ten_hang_hoa, ten_don_vi, gia_ban)
     * vào các phần tử trong list_hang_hoa (level 1 và trong options).
     */
    private function attachHangHoaInfo($data){
        $hangHoaModel = new HangHoa();

        // --- TRƯỜNG HỢP list_hang_hoa ở level 1 (many_options = false) ---
        if (!empty($data['list_hang_hoa']) ) {
            
            foreach ($data['list_hang_hoa'] as &$item) {
                
                $hh = $hangHoaModel->getHangHoa($item['id_hang_hoa'], 'part-detail');
                if (!is_array($hh)) {
                    continue; 
                }

                // Gắn thêm thông tin từ hàng hoá
                $item['id_hang_hoa'] = $hh['_id']; 
                $item['ten_hang_hoa'] = $hh['ten_hang_hoa'] ?? null;
                $item['ten_don_vi']   = $hh['ten_don_vi']   ?? null;
                $item['gia_ban']      = $hh['gia_ban']      ?? null;
            }
            unset($item);
        }

        // --- TRƯỜNG HỢP list_hang_hoa nằm trong từng option (many_options = true) ---
        if (!empty($data['options'])) {
            
            foreach ($data['options'] as &$option) {
                if (empty($option['list_hang_hoa'])) {
                    continue;
                }

                
                foreach ($option['list_hang_hoa'] as &$item) {
                    $hh = $hangHoaModel->getHangHoa($item['id_hang_hoa'], 'part-detail');
                    
                    if (empty($hh)) {
                        continue;
                    }

                    $item['id_hang_hoa'] = $hh['_id'];
                    $item['ten_hang_hoa'] = $hh['ten_hang_hoa'] ?? null;
                    $item['ten_don_vi']   = $hh['ten_don_vi']   ?? null;
                    $item['gia_ban']      = $hh['gia_ban']      ?? null;
                }
                
                unset($item);
            }
            unset($option);
        }
        return $data;
    }

    function getOption($data, $optionName) {

        // Kiểm tra xem dữ liệu có chứa mảng options không
        if (!isset($data['options'])) {
            return null;
        }

        // Lặp qua từng option
        foreach ($data['options'] as $opt) {

            // Nếu tên option trùng với tên bạn cần
            if (isset($opt['ten_option']) && $opt['ten_option'] === $optionName) {

                // Trả về dữ liệu theo format bạn muốn
                return [
                    "_id"           => $data["_id"],
                    "ten_dich_vu"   => $data["ten_dich_vu"] ." - " . $opt["ten_option"],
                    "ten_don_vi"    => $data["ten_don_vi"],

                    // Lấy giá option (hỗ trợ cả gia_option & gia_otion nếu có typo)
                    "gia_option"    => $opt["gia_option"] ?? ($opt["gia_otion"] ?? null),

                    // Lấy list hàng hóa nếu có
                    "list_hang_hoa" => $opt["list_hang_hoa"] ?? []
                ];
            }
        }

        // Nếu không tìm thấy option phù hợp
        return null;
    }



    // Thêm dịch vụ mới
    public function addDichVu($data) {

        $data = $this->processData($data);
    
        $insertResult = $this->collection->insertOne($data);
        return (string) $insertResult->getInsertedId(); 

    }

    // Cập nhật thông tin dịch vụ
    public function updateDichVu($id, $data) {

        $data =  $this->processData( $data);

        //Sử dụng replaceOne để ghi đè mọi dữ liệu 
        $updateResult = $this->collection->replaceOne(
            ['_id' => new MongoDB\BSON\ObjectId($id)],
            $data
        );

        return $updateResult->getModifiedCount(); // Số bản ghi bị thay đổi
    }

    //get Dich Vu theo _id
    public function getDichVu($id, $optionName = ""){
        $cursor = $this->collection->findOne([
            '_id' => new MongoDB\BSON\ObjectId($id)
        ]);

        if (!$cursor) {
            return null;
        }

        $data = $cursor->getArrayCopy();

        $data = $this->attachHangHoaInfo($data);

        //Nếu chỉ lấy bản nhỏ gọn của dịch vụ
        if (!empty($optionName)){
            $data = $this->getOption($data, $optionName);
        }


        return JSONHelper::removeOid($data);
    }

    //Function Get All Dịch vụ 
    public function getAllDichVu($data){
        // Lấy page
        $page = isset($data["page"]) ? (int)$data["page"] : 1;
        if ($page < 1) $page = 1;

        // Lấy kw
        $kw = isset($data["kw"]) ? trim($data["kw"]) : "";

        // Lấy phân loại (exact match)
        $phan_loai = isset($data["ten_phan_loai"]) ? trim($data["ten_phan_loai"]) : "";

        // Giới hạn & skip
        $limit = 15;
        $skip = ($page - 1) * $limit;

        // Tạo filter
        $filter = [];

        // Lọc theo keyword
        if ($kw !== "") {
            $filter["ten_dich_vu"] = new MongoDB\BSON\Regex($kw, "i");
        }

        // Lọc theo phân loại — EXACT MATCH
        if ($phan_loai !== "") {
            $filter["ten_phan_loai"] = $phan_loai;  // so sánh tuyệt đối
        }

        // Truy vấn MongoDB
        $cursor = $this->collection->find(
            $filter,
            [
                "limit" => $limit,
                "skip"  => $skip,
                "sort"  => ["_id" => -1],
                "projection" => [
                    "_id"           => 1,
                    "ten_dich_vu"   => 1,
                    "ten_phan_loai" => 1,
                    "ten_don_vi"    => 1
                ]
            ]
        );

        $items = iterator_to_array($cursor);

        // Đếm tổng bản ghi
        $totalItems = $this->collection->countDocuments($filter);
        $totalPages = (int)ceil($totalItems / $limit);

        // Trả về kết quả
        return [
            "list_dich_vu" => JSONHelper::removeOid($items),
            "pagination" => [
                "page"        => $page,
                "total_items" => $totalItems,
                "total_pages" => $totalPages
            ]
        ];
    }

    // DELETE: Xóa dịch vụ
    public function deleteDichVu($id) {
        $deleteResult = $this->collection->deleteOne(['_id' => new MongoDB\BSON\ObjectId($id)]);
        return $deleteResult->getDeletedCount(); // Trả về số bản ghi đã bị xóa
    }

    // AutoComplete Dịch vụ với option 
    public function autoCompleteDichVu($searchQuery) {
        // Nếu chuỗi tìm kiếm rỗng thì không làm gì
        if ($searchQuery == "")
            return null;

        // Sử dụng Regex để tìm tên dịch vụ, không phân biệt hoa/thường
        $regex = new MongoDB\BSON\Regex($searchQuery, 'i');

        // Tìm dịch vụ theo tên
        $cursor = $this->collection->aggregate([
            [
                '$match' => [
                    'ten_dich_vu' => $regex
                ]
            ],
            [
                // Lấy các trường cần thiết: _id, tên dịch vụ và options
                '$project' => [
                    '_id'         => 1,
                    'ten_dich_vu' => 1,
                    'gia_ban' => 1,
                    'options'     => 1
                ]
            ],
            [
                // Giới hạn số lượng gợi ý autocomplete
                '$limit' => 5
            ]
        ]);

        // Chuyển kết quả về mảng PHP và loại bỏ ObjectId (chuyển sang string)
        $raw = JSONHelper::removeOid(iterator_to_array($cursor));

        $result = [];

        foreach ($raw as $item) {
            // Dữ liệu cơ bản luôn có: _id và tên dịch vụ
            $row = [
                '_id'         => $item['_id'] ?? null,
                'ten_dich_vu' => $item['ten_dich_vu'] ?? null
            ];

            // Nếu có options thì gom toàn bộ ten_option lại
            if (isset($item['options']) ) {
                $optionsData = [];

                foreach ($item['options'] as $opt) {

                    // Bỏ qua option rỗng
                    if (isset($opt['ten_option']) && $opt['ten_option'] !== "") {
                        $optionsData[] = [
                            'ten_option' => $opt['ten_option']
                        ];
                    }
                }

                if (!empty($optionsData)) {
                    $row['options'] = $optionsData;
                }
            }
            $result[] = $row;
    }

    return $result;
    }
}

?>