<?php

// ===== CORS ALLOW ALL =====
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

#############################################################################

require_once 'vendor/autoload.php';
require_once 'controller/ResponseHelper.php';

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\File\UploadedFile;

// ---- Cấu hình giới hạn ----
const MAX_IMG_FILES   = 10;                 // tối đa 10 ảnh
const MAX_IMG_SIZE    = 2 * 1024 * 1024;    // 2MB/ảnh
const MAX_FILE_SIZE   = 1 * 1024 * 1024;    // 1MB/file thường

$ALLOWED_FILE_MIMES = [
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', // xlsx
    'application/vnd.ms-excel',                                          // xls
    'text/csv',
    'application/csv'
];

// ---- Lấy Request ----
$request = Request::createFromGlobals();

// Chỉ cho POST
if ($request->getMethod() !== 'POST') {
    ResponseHelper::responseError('Method not allowed, use POST.');
    exit;
}

// Lấy type từ POST body (multipart/form-data hoặc x-www-form-urlencoded)
$type = (string) $request->request->get('type', '');
if ($type === '') {
    ResponseHelper::responseError("Thiếu trường POST 'type'. Giá trị hợp lệ: 'img' hoặc 'file'.");
    exit;
}
$type = strtolower($type);
if (!in_array($type, ['img', 'file_excel'], true)) {
    ResponseHelper::responseError("Giá trị 'type' không hợp lệ. Chỉ nhận 'img' hoặc 'file'.");
    exit;
}

$isImg   = ($type === 'img');
$subDir  = $isImg ? 'img' : 'file_excel';
$rootDir = dirname(__DIR__); // Thư mục gốc project (nếu file đặt trong public/)
$basePath = $rootDir . "/uploads/{$subDir}";

// Tạo thư mục nếu chưa có
if (!is_dir($basePath)) {
    @mkdir($basePath, 0775, true);
    if (!is_dir($basePath)) {
        ResponseHelper::responseError('Không tạo được thư mục lưu file.');
        exit;
    }
}

// =============== XỬ LÝ ẢNH ===============
if ($isImg) {
    /** @var UploadedFile[]|null $files */
    $files = $request->files->get('files');

    if (!$files || !is_array($files)) {
        ResponseHelper::responseError("Thiếu trường 'files[]' (multipart/form-data).");
        exit;
    }
    if (count($files) > MAX_IMG_FILES) {
        ResponseHelper::responseError("Quá số lượng ảnh cho phép: tối đa " . MAX_IMG_FILES . " file.");
        exit;
    }

    $saved = [];
    foreach ($files as $idx => $file) {
        if (!$file instanceof UploadedFile) {
            ResponseHelper::responseError("Phần tử files[$idx] không hợp lệ.");
            exit;
        }
        if ($file->getError() !== UPLOAD_ERR_OK) {
            ResponseHelper::responseError("Upload ảnh thất bại ở files[$idx], mã lỗi: " . $file->getError());
            exit;
        }

        // Kiểm tra mime
        $mime = (string) $file->getMimeType();
        if (strpos($mime, 'image/') !== 0) {
            ResponseHelper::responseError("Chỉ cho phép file ảnh (jpg, png, webp, ...) ở files[$idx].");
            exit;
        }
        // Kiểm tra dung lượng
        if ($file->getSize() > MAX_IMG_SIZE) {
            ResponseHelper::responseError("Ảnh files[$idx] vượt quá " . (int)(MAX_IMG_SIZE/1024/1024) . "MB.");
            exit;
        }

        $saved[] = store_uploaded_file($file, $basePath, $subDir); // ['filename'=>..., 'path'=>...]
    }

    // 🔹 Chỉ trả về danh sách URL
    ResponseHelper::responseSuccess(array_column($saved, 'path'));
    exit;
}

// =============== XỬ LÝ FILE THƯỜNG (Excel/CSV) ===============
/** @var UploadedFile|array|null $file */
$file = $request->files->get('file');

if (is_array($file)) {
    ResponseHelper::responseError("Chỉ được upload 1 file duy nhất (field 'file').");
    exit;
}
if (!$file instanceof UploadedFile) {
    ResponseHelper::responseError("Thiếu trường 'file' (multipart/form-data).");
    exit;
}
if ($file->getError() !== UPLOAD_ERR_OK) {
    ResponseHelper::responseError("Upload file thất bại, mã lỗi: " . $file->getError());
    exit;
}
if ($file->getSize() > MAX_FILE_SIZE) {
    ResponseHelper::responseError("File vượt quá " . (int)(MAX_FILE_SIZE/1024/1024) . "MB.");
    exit;
}

$mime = (string) $file->getMimeType();
if (!in_array($mime, $ALLOWED_FILE_MIMES, true)) {
    ResponseHelper::responseError("Chỉ cho phép file xlsx, xls, csv.");
    exit;
}

$saved = store_uploaded_file($file, $basePath, $subDir);

// 🔹 Trả về list 1 phần tử (URL duy nhất)
ResponseHelper::responseSuccess([$saved['path']]);
exit;

// ================== HÀM LƯU FILE ==================
/**
 * Lưu file upload vào thư mục đích và trả về tên + đường dẫn public.
 *
 * @return array{filename:string,path:string}
 */
function store_uploaded_file(UploadedFile $file, string $basePath, string $subDir): array
{
    $origName = pathinfo((string)$file->getClientOriginalName(), PATHINFO_FILENAME);
    $ext = $file->guessExtension() ?: $file->getClientOriginalExtension() ?: 'bin';
    $safe = preg_replace('~[^a-z0-9\-]+~i', '-', (string)$origName) ?: 'file';
    $rand = substr(bin2hex(random_bytes(6)), 0, 12);
    $name = sprintf('%s-%s.%s', $safe, $rand, $ext);

    // move() sẽ đổi tên + di chuyển an toàn
    $file->move($basePath, $name);

    return [
        'filename' => $name,
        'path'     => "/uploads/{$subDir}/{$name}",
    ];
}
